package dev.amble.ait.core.tardis.control.sound;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.concurrent.atomic.AtomicReference;
import net.minecraft.class_2960;
import net.minecraft.class_3414;
import net.minecraft.class_3417;
import net.minecraft.class_7923;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.mojang.serialization.Codec;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.amble.lib.api.Identifiable;
import dev.amble.ait.AITMod;
import dev.amble.ait.core.AITSounds;
import dev.amble.ait.core.tardis.control.Control;
import dev.amble.ait.data.schema.console.ConsoleTypeSchema;
import dev.amble.ait.registry.impl.ControlRegistry;
import dev.amble.ait.registry.impl.console.ConsoleRegistry;

/**
 * Represents a sound that is played when a control is used
 * @param controlId The identifier of the control that this sound is for
 * @param consoleId The identifier of the console that this sound is for
 * @param successId The identifier of the sound that is played when the control is successful
 * @param altId The identifier of the sound that is played when the control fails OR a value is switched
 * @see Control
 * @see ConsoleTypeSchema
 * @author duzo
 */
public record ControlSound(class_2960 controlId, class_2960 consoleId, class_2960 successId, class_2960 altId) implements Identifiable {
    public static final Codec<ControlSound> CODEC = RecordCodecBuilder.create(instance -> instance.group(
            class_2960.field_25139.fieldOf("control").forGetter(ControlSound::controlId),
            class_2960.field_25139.fieldOf("console").forGetter(ControlSound::consoleId),
            class_2960.field_25139.fieldOf("success_sound").forGetter(ControlSound::successId),
            class_2960.field_25139.optionalFieldOf("alt_sound", class_3417.field_42593.method_14833()).forGetter(ControlSound::altId)
    ).apply(instance, ControlSound::new));

    @Override
    public class_2960 id() {
        return mergeIdentifiers(controlId, consoleId);
    }

    public ConsoleTypeSchema console() {
        return ConsoleRegistry.getInstance().get(this.consoleId());
    }
    public Control control() {
        return ControlRegistry.REGISTRY.method_10223(this.controlId());
    }
    public class_3414 sound(Control.Result result) {
        return !result.isAltSound() ? this.successSound() : this.altSound();
    }

    public class_3414 successSound() {
        class_3414 sfx = class_7923.field_41172.method_10223(this.successId());

        if (sfx == null) {
            AITMod.LOGGER.error("Unknown success sound event: {} in control sfx {}", this.successId(), this.id());
            sfx = AITSounds.ERROR;
        }

        return sfx;
    }
    public class_3414 altSound() {
        class_3414 sfx = class_7923.field_41172.method_10223(this.altId());

        if (sfx == null || this.altId() == class_3417.field_42593.method_14833()) {
            AITMod.LOGGER.error("Unknown alt sound event: {} in control sfx {}", this.altId(), this.id());
            sfx = successSound();
        }

        return sfx;
    }

    public static ControlSound forFallback(class_2960 controlId, class_3414 success, class_3414 alt) {
        return new ControlSound(controlId, AITMod.id("fallback"), success.method_14833(), alt.method_14833());
    }

    /**
     * Merges the two identifiers into one
     * Example
     * controlId - ait:monitor
     * consoleId - ait:hartnell
     * return - ait:hartnell/ait/monitor
     * @param controlId id of the control
     * @param consoleId id of the console
     * @return Merged identifier
     */
    public static class_2960 mergeIdentifiers(class_2960 controlId, class_2960 consoleId) {
        return class_2960.method_43902(consoleId.method_12836(), consoleId.method_12832() + "/" + controlId.method_12836() + "/" + controlId.method_12832());
    }

    public static ControlSound fromInputStream(InputStream stream) {
        return fromJson(JsonParser.parseReader(new InputStreamReader(stream)).getAsJsonObject());
    }

    public static ControlSound fromJson(JsonObject json) {
        AtomicReference<ControlSound> created = new AtomicReference<>();

        CODEC.decode(JsonOps.INSTANCE, json).get().ifLeft(var -> created.set(var.getFirst())).ifRight(err -> {
            created.set(null);
            AITMod.LOGGER.error("Error decoding datapack console variant: {}", err);
        });

        return created.get();
    }
}
