package dev.amble.ait.core.util;

import java.util.function.Consumer;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_2791;
import net.minecraft.class_2902;
import net.minecraft.class_3218;
import net.minecraft.class_3542;
import net.minecraft.class_5250;
import dev.amble.lib.data.CachedDirectedGlobalPos;
import dev.drtheo.queue.api.ActionQueue;
import dev.drtheo.queue.api.util.Value;
import dev.drtheo.scheduler.api.TimeUnit;
import dev.drtheo.scheduler.api.common.TaskStage;
import org.jetbrains.annotations.Nullable;

public class SafePosSearch {

    private static final int SAFE_RADIUS = 3;

    public static void wrapSafe(CachedDirectedGlobalPos globalPos, Kind vSearch,
                                boolean hSearch, Consumer<CachedDirectedGlobalPos> posConsumer) {
        Value<class_2338> ref = new Value<>(null);
        ActionQueue queue = findSafe(globalPos, vSearch, hSearch, ref);

        if (queue != null) {
            queue.thenRun(() -> {
                CachedDirectedGlobalPos resultPos = globalPos;

                if (ref.value != null)
                    resultPos = resultPos.pos(ref.value);

                posConsumer.accept(resultPos);
            }).execute();
        } else {
            posConsumer.accept(globalPos);
        }
    }

    /**
     * @return {@literal null} when the position is already safe, {@link ActionQueue} otherwise.
     */
    @Nullable public static ActionQueue findSafe(CachedDirectedGlobalPos globalPos,
                                       Kind vSearch, boolean hSearch, Value<class_2338> ref) {
        class_3218 world = globalPos.getWorld();
        class_2338 pos = globalPos.getPos();

        final class_2791 chunk = globalPos.getWorld().method_22350(pos);

        if (isSafe(chunk, pos))
            return null;

        ActionQueue queue = new ActionQueue();

        if (hSearch) {
            queue = findSafeXZ(queue, ref, world, pos, SAFE_RADIUS).thenRun(() -> {
                if (ref.value != null)
                    globalPos.pos(ref.value);
            });
        }

        return switch (vSearch) {
            case CEILING -> findSafeCeiling(queue, ref, world, pos);
            case FLOOR -> findSafeFloor(queue, ref, world, pos);
            case MEDIAN -> findSafeMedian(queue, ref, world, pos);
            case NONE -> queue;
        };
    }

    private static ActionQueue findSafeCeiling(ActionQueue queue, Value<class_2338> result, class_3218 world, class_2338 original) {
        return queue.thenRun(() -> {
            if (result.value != null)
                return;

            int y = world.method_22350(original).method_12005(class_2902.class_2903.field_13203,
                    original.method_10263() & 15, original.method_10260() & 15) + 1;

            result.value = original.method_33096(y);
        });
    }

    private static ActionQueue findSafeFloor(ActionQueue queue, Value<class_2338> result, class_3218 world, class_2338 original) {
        final SafeFloorHolder holder = new SafeFloorHolder(world, original);

        return queue.thenRunSteps(() -> {
            if (result.value != null)
                return true;

            Iter state = holder.checkAndAdvance();

            if (state == Iter.SUCCESS)
                result.value = holder.cursor;

            return state != Iter.CONTINUE;
        }, TaskStage.startWorldTick(world), TimeUnit.TICKS, 1, 3);
    }

    private static ActionQueue findSafeMedian(ActionQueue queue, Value<class_2338> result, class_3218 world, class_2338 original) {
        final SafeMedianHolder holder = new SafeMedianHolder(world, original);

        return queue.thenRunSteps(() -> {
            if (result.value != null)
                return true;

            DoubleIter state = holder.checkAndAdvance();

            if (state == DoubleIter.SUCCESS_A) {
                result.value = holder.upCursor;
            } else if (state == DoubleIter.SUCCESS_B) {
                result.value = holder.downCursor;
            }

            return state != DoubleIter.CONTINUE;
        }, TaskStage.startWorldTick(world), TimeUnit.TICKS, 1, 3);
    }

    private static ActionQueue findSafeXZ(ActionQueue queue, Value<class_2338> result, class_3218 world, class_2338 original, int radius) {
        class_2338.class_2339 pos = original.method_25503();

        int minX = pos.method_10263() - radius;
        int maxX = pos.method_10263() + radius;

        int minZ = pos.method_10260() - radius;
        int maxZ = pos.method_10260() + radius;

        final SafeXZHolder holder = new SafeXZHolder(world, pos, maxX, maxZ, minX, minZ);

        return queue.thenRunSteps(() -> {
            Iter state = holder.checkAndAdvance();

            if (state == Iter.SUCCESS)
                result.value = holder.pos.method_10062();

            return state != Iter.CONTINUE;
        }, TaskStage.startWorldTick(world), TimeUnit.TICKS, 1, 3); // every tick, while the taken time is less than 3ms (1tick = 50ms, 2/50 of a tick, which is 4%)
    }

    @SuppressWarnings("deprecation")
    private static boolean isSafe(class_2791 chunk, class_2338 pos) {
        class_2680 floor = chunk.method_8320(pos.method_10074());

        if (!floor.method_51366())
            return false;

        class_2680 curUp = chunk.method_8320(pos);
        class_2680 aboveUp = chunk.method_8320(pos.method_10084());

        return !curUp.method_51366() && !aboveUp.method_51366();
    }

    @SuppressWarnings("deprecation")
    private static boolean isSafe(class_2680 floor, class_2680 block1, class_2680 block2) {
        return floor.method_51366() && !block1.method_51366() && !block2.method_51366();
    }

    static class SafeXZHolder {
        int x;
        int z;
        class_2791 prevChunk;
        final class_1937 world;
        final class_2338.class_2339 pos;
        final int maxX;
        final int maxZ;
        final int minX;

        public SafeXZHolder(class_1937 world, class_2338.class_2339 pos, int maxX, int maxZ, int minX, int minZ) {
            this.world = world;
            this.pos = pos;
            this.maxX = maxX;
            this.maxZ = maxZ;
            this.minX = minX;
            this.x = minX;
            this.z = minZ;
        }

        public Iter checkAndAdvance() {
            if (z >= maxZ)
                return Iter.FAIL;

            if (x >= maxX) {
                x = minX;
                z += 1;

                return Iter.CONTINUE;
            }

            pos.method_33097(x).method_33099(z);

            class_1923 tempPos = new class_1923(pos);
            if (prevChunk == null || !prevChunk.method_12004().equals(tempPos))
                prevChunk = world.method_8497(tempPos.field_9181, tempPos.field_9180);

            if (isSafe(prevChunk, pos))
                return Iter.SUCCESS;

            x += 1;
            return Iter.CONTINUE;
        }

        public class_2338 pos() {
            return pos.method_10062();
        }
    }

    static class SafeFloorHolder {
        class_2338 cursor;
        class_2680 floor;
        class_2680 current;
        class_2680 above;

        final class_2791 chunk;
        final int maxY;

        public SafeFloorHolder(class_1937 world, class_2338 pos) {
            this.chunk = world.method_22350(pos);
            this.maxY = chunk.method_31600();

            int minY = chunk.method_31607();
            this.cursor = pos.method_33096(minY + 2);

            this.floor = chunk.method_8320(cursor.method_10074());
            this.current = chunk.method_8320(cursor);
            this.above = chunk.method_8320(cursor.method_10084());
        }

        public Iter checkAndAdvance() {
            if (cursor.method_10264() >= maxY)
                return Iter.FAIL;

            if (isSafe(floor, current, above))
                return Iter.SUCCESS;

            cursor = cursor.method_10084();

            floor = current;
            current = above;
            above = chunk.method_8320(cursor);

            return Iter.CONTINUE;
        }
    }

    static class SafeMedianHolder {

        class_2338 upCursor;
        class_2680 floorUp;
        class_2680 curUp;
        class_2680 aboveUp;

        class_2338 downCursor;
        class_2680 floorDown;
        class_2680 curDown;
        class_2680 aboveDown;

        final class_2791 chunk;

        public SafeMedianHolder(class_1937 world, class_2338 pos) {
            this.chunk = world.method_22350(pos);

            this.upCursor = pos.method_10084();
            this.floorUp = chunk.method_8320(upCursor.method_10074());
            this.curUp = chunk.method_8320(upCursor);
            this.aboveUp = chunk.method_8320(upCursor.method_10084());

            this.downCursor = pos.method_10074();
            this.floorDown = chunk.method_8320(downCursor.method_10074());
            this.curDown = chunk.method_8320(downCursor);
            this.aboveDown = chunk.method_8320(downCursor.method_10084());
        }

        public DoubleIter checkAndAdvance() {
            boolean canGoUp = upCursor.method_10264() < chunk.method_31600();
            boolean canGoDown = downCursor.method_10264() > chunk.method_31607();

            if (!canGoUp && !canGoDown)
                return DoubleIter.FAIL;

            if (canGoUp) {
                if (isSafe(floorUp, curUp, aboveUp)) {
                    upCursor = upCursor.method_10074();
                    return DoubleIter.SUCCESS_A;
                }

                upCursor = upCursor.method_10084();

                floorUp = curUp;
                curUp = aboveUp;
                aboveUp = chunk.method_8320(upCursor);
            }

            if (canGoDown) {
                if (isSafe(floorDown, curDown, aboveDown)) {
                    downCursor = downCursor.method_10084();
                    return DoubleIter.SUCCESS_B;
                }

                downCursor = downCursor.method_10074();

                curDown = aboveDown;
                aboveDown = floorDown;
                floorDown = chunk.method_8320(downCursor);
            }

            return DoubleIter.CONTINUE;
        }
    }

    enum Iter {
        SUCCESS,
        FAIL,
        CONTINUE
    }

    enum DoubleIter {
        SUCCESS_A,
        SUCCESS_B,
        FAIL,
        CONTINUE
    }

    public enum Kind implements class_3542 {
        NONE {
            @Override
            public Kind next() {
                return FLOOR;
            }
        },
        FLOOR {
            @Override
            public Kind next() {
                return CEILING;
            }
        },
        CEILING {
            @Override
            public Kind next() {
                return MEDIAN;
            }
        },
        MEDIAN {
            @Override
            public Kind next() {
                return NONE;
            }
        };

        @Override
        public String method_15434() {
            return toString();
        }

        public class_5250 text() {
            return class_2561.method_43471("message.ait.control.ylandtype." + this.method_15434().toLowerCase());
        }

        public abstract Kind next();
    }
}
