package dev.amble.ait.data.landing;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_1923;
import net.minecraft.class_2338;
import com.google.common.collect.ImmutableCollection;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.amble.lib.util.ServerLifecycleHooks;
import org.jetbrains.annotations.Nullable;
import dev.amble.ait.core.tardis.manager.ServerTardisManager;

public class LandingPadRegion {

    public static final Codec<LandingPadRegion> CODEC = RecordCodecBuilder.create(instance -> instance.group(
            Codec.LONG.fieldOf("chunk").forGetter(o -> o.getChunk().method_8324()),
            Codec.INT.fieldOf("y").forGetter(LandingPadRegion::getDefaultY),
            Codec.list(LandingPadSpot.CODEC).fieldOf("spots").forGetter(o -> o.spots),
            Codec.STRING.fieldOf("code").forGetter(LandingPadRegion::getLandingCode)
    ).apply(instance, LandingPadRegion::create));

    private static final int CHUNK_LENGTH = 16;
    private static final int MAX_SPOTS = (CHUNK_LENGTH * CHUNK_LENGTH) / 4;
    private static final int MAX_PER_ROW = CHUNK_LENGTH / 2;

    private final class_1923 chunk;
    private final List<LandingPadSpot> spots;

    private final int defaultY;
    private String landingCode;

    private static LandingPadRegion create(long chunk, int y, List<LandingPadSpot> spots, String landingCode) {
        if (spots instanceof ImmutableCollection<?>) {
            spots = new ArrayList<>(spots);
        }

        return new LandingPadRegion(new class_1923(chunk), y, spots, landingCode);
    }

    private LandingPadRegion(class_1923 chunk, int y, List<LandingPadSpot> spots, String landingCode) {
        this.chunk = chunk;
        this.spots = spots;
        this.landingCode = landingCode;

        this.defaultY = y;

        if (spots.isEmpty())
            this.createAllSpots();
    }

    public LandingPadRegion(class_1923 pos, int y, String landingCode) {
        this(pos, y, new ArrayList<>(), landingCode);
    }

    public @Nullable LandingPadSpot getFreeSpot() {
        for (LandingPadSpot spot : this.spots) {
            if (spot.isOccupied())
                continue;

            return spot;
        }

        return null; // unreachable
    }

    private LandingPadSpot createSpot() {
        if (this.spots.isEmpty())
            return new LandingPadSpot(new class_2338(
                    this.chunk.method_8326() + 1, this.defaultY, this.chunk.method_8328() + 1
            ));

        float rowCount = ((float) this.spots.size() / MAX_PER_ROW);
        boolean isNewRow = rowCount == Math.round(rowCount);
        class_2338 lastPos = this.spots.get(this.spots.size() - 1).getPos();

        if (!isNewRow)
            return new LandingPadSpot(lastPos.method_10069(2, 0, 0));

        return new LandingPadSpot(new class_2338(this.spots.get(0).getPos().method_10263(),
                this.defaultY, lastPos.method_10260() + 2));
    }

    private LandingPadSpot generateSpot() {
        LandingPadSpot created = this.createSpot();
        this.spots.add(created);

        return created;
    }

    public List<LandingPadSpot> getSpots() {
        return this.spots;
    }

    public int getDefaultY() {
        return defaultY;
    }

    public class_1923 getChunk() {
        return chunk;
    }

    public String getLandingCode() {
        return landingCode;
    }

    public void setLandingCode(String string) {
        landingCode = string;
    }

    private boolean isFull() {
        return this.getFreeSpot() == null; // all spots must be occupied if none is found
    }

    private void createAllSpots() {
        int toCreate = MAX_SPOTS - this.spots.size();

        for (int i = 0; i < toCreate; i++) {
            this.generateSpot();
        }
    }

    public Optional<LandingPadSpot> getSpotAt(class_2338 pos) {
        // is in this chunk?
        if (this.chunk.method_8324() != new class_1923(pos).method_8324())
            return Optional.empty();

        // is in this region?
        for (LandingPadSpot spot : this.spots) {
            class_2338 p = spot.getPos();
            if (p.method_10263() == pos.method_10263() && p.method_10260() == pos.method_10260()) // ignore Y
                return Optional.of(spot);
        }

        // not found
        return Optional.empty();
    }

    public LandingPadSpot createSpotAt(class_2338 pos) {
        LandingPadSpot existing = this.getSpotAt(pos).orElse(null);

        if (existing != null) return existing;

        LandingPadSpot created = new LandingPadSpot(pos);
        this.spots.add(created);

        return created;
    }

    public void removeSpotAt(class_2338 pos) {
        LandingPadSpot existing = this.getSpotAt(pos).orElse(null);
        if (existing == null) return;

        this.removeSpot(existing);
    }

    private void removeSpot(LandingPadSpot spot) {
        this.spots.remove(spot);

        if (!spot.isOccupied()) return;

        // should the tardis take off and find a new spot?
        ServerTardisManager.getInstance().getTardis(ServerLifecycleHooks.get(), spot.getReference().getId(), tardis -> tardis.landingPad().release());
    }

    @Override
    public String toString() {
        return "LandingPadRegion{" +
                "chunk=" + chunk +
                ", spots=" + spots +
                ", defaultY=" + defaultY +
                '}';
    }
}
