package dev.amble.ait.registry.impl;

import java.util.HashMap;
import java.util.List;

import dev.amble.ait.client.AITModClient;
import net.fabricmc.fabric.api.event.registry.FabricRegistryBuilder;
import net.fabricmc.fabric.api.message.v1.ServerMessageEvents;
import net.minecraft.class_1799;
import net.minecraft.class_2370;
import net.minecraft.class_2378;
import net.minecraft.class_2556;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.class_3414;
import net.minecraft.class_5321;
import net.minecraft.class_7471;
import dev.amble.ait.AITMod;
import dev.amble.ait.core.AITSounds;
import dev.amble.ait.core.handles.HandlesResponse;
import dev.amble.ait.core.handles.HandlesSound;
import dev.amble.ait.core.item.HandlesItem;
import dev.amble.ait.core.tardis.ServerTardis;
import dev.amble.ait.core.tardis.Tardis;
import dev.amble.ait.core.tardis.control.impl.SecurityControl;
import dev.amble.ait.core.tardis.handler.travel.TravelHandlerBase;
import dev.amble.ait.core.world.TardisServerWorld;

/**
 * Registry for Handles responses.
 * This is using Minecraft Registries, so just call HandlesResponseRegistry.register in your mod initialization.
 * @author james
 */
public class HandlesResponseRegistry {
    public static final class_2370<HandlesResponse> REGISTRY = FabricRegistryBuilder
            .createSimple(class_5321.<HandlesResponse>method_29180(AITMod.id("handles")))
            .buildAndRegister();
    private static HashMap<String, HandlesResponse> COMMANDS_CACHE;
    public static HandlesResponse DEFAULT;

    public static HandlesResponse register(HandlesResponse schema) {
        COMMANDS_CACHE = null;

        return class_2378.method_10230(REGISTRY, schema.id(), schema);
    }

    public static HandlesResponse get(String command) {
        if (COMMANDS_CACHE == null) {
            fillCommands();
        }
        HandlesResponse found = COMMANDS_CACHE.get(command);

        if (found != null) {
            return found;
        }

        int minDistance = Integer.MAX_VALUE;
        HandlesResponse closest = null;
        for (String key : COMMANDS_CACHE.keySet()) {
            int distanc = COMMANDS_CACHE.get(key).distance(key, command);
            if (distanc < minDistance) {
                minDistance = distanc;
                closest = COMMANDS_CACHE.get(key);
            }
        }

        if (closest != null && minDistance <= AITModClient.CONFIG.handlesLevenshteinDistance) {
            return closest;
        }

        return DEFAULT;
    }

    private static void fillCommands() {
        COMMANDS_CACHE = new HashMap<>();
        for (HandlesResponse response : REGISTRY) {
            for (String command : response.getCommandWords()) {
                COMMANDS_CACHE.put(command, response);
            }
        }
    }

    public static void init() {
        ServerMessageEvents.ALLOW_CHAT_MESSAGE.register(HandlesResponseRegistry::onChatMessage);

        DEFAULT = register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                return failure(source);
            }

            @Override
            public class_3414 failureSound() {
                return AITMod.RANDOM.nextBoolean() ? AITSounds.HANDLES_PLEASE_ASK_AGAIN : AITSounds.HANDLES_PARDON;
            }

            @Override
            public List<String> getCommandWords() {
                return List.of();
            }

            @Override
            public class_2960 id() {
                return AITMod.id("default");
            }
        });

        register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                sendChat(player, getHelpText());
                return success(source);
            }

            private class_2561 getHelpText() {
                return class_2561.method_43470("Available Commands: " + String.join(", ", COMMANDS_CACHE.keySet()));
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("help");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("help");
            }
        });

        register(new HandlesResponse() {
            private static final List<String> JOKES = List.of(
                    "Why did the Dalek apply for a job? It wanted to EX-TER-MINATE its competition!",
                    "How many Time Lords does it take to change a light bulb? None, they just change the timeline.",
                    "Why does the TARDIS always win hide-and-seek? Because it’s in another dimension!",
                    "What do you call a Time Lord with no time? A Lord!",
                    "Why was the TARDIS always calm? Because it’s bigger on the inside."
            );

            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                sendChat(player, getRandomJoke());
                return success(source);
            }

            private class_2561 getRandomJoke() {
                return class_2561.method_43470(JOKES.get(AITMod.RANDOM.nextInt(JOKES.size()) - 1));
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("tell me a joke");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("joke");
            }
        });

        register(new HandlesResponse() {
            private static final List<String> FUN_FACTS = List.of(
                    "The first TARDIS was actually painted green!",
                    "Gallifrey has two suns and an orange sky!",
                    "Handles once saved the Doctor’s life by solving a centuries-old riddle."
            );

            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                sendChat(player, getRandomFunFact());
                return success(source);
            }

            private class_2561 getRandomFunFact() {
                return class_2561.method_43470(FUN_FACTS.get(AITMod.RANDOM.nextInt(FUN_FACTS.size()) - 1));
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("tell me a fun fact");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("fun_fact");
            }
        });

        register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                if (tardis.travel().inFlight()) {
                    sendChat(player, class_2561.method_43470("The TARDIS is already in flight.."));
                    return failure(source);
                }

                tardis.travel().dematerialize();
                sendChat(player, class_2561.method_43470("Initiating dematerialization sequence."));
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("dematerialize", "take off");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("dematerialize");
            }
        });

        register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                if (!tardis.travel().inFlight()) {
                    sendChat(player, class_2561.method_43470("The TARDIS is not in flight."));
                    return failure(source);
                }

                tardis.travel().rematerialize();
                sendChat(player, class_2561.method_43470("Rematerializing."));
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("rematerialize", "land");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("rematerialize");
            }
        });

        register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                if (tardis.door().locked()) {
                    sendChat(player, class_2561.method_43470("Doors already locked"));
                    return failure(source);
                }

                tardis.door().setLocked(true);
                sendChat(player, class_2561.method_43470("Locking door."));
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("lock", "lock door");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("lock");
            }
        });

        register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                if (!tardis.door().locked()) {
                    sendChat(player, class_2561.method_43470("Doors already unlocked"));
                    return failure(source);
                }

                tardis.door().setLocked(false);
                sendChat(player, class_2561.method_43470("Unlocking door."));
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("unlock", "unlock door");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("unlock");
            }
        });

        register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                if (!tardis.waypoint().hasWaypoint()) {
                    sendChat(player, class_2561.method_43470("There is no waypoint set."));
                    return failure(source);
                }

                sendChat(player, class_2561.method_43470("Setting course for waypoint."));
                tardis.waypoint().loadWaypoint();
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("go to waypoint", "travel to waypoint");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("travel_waypoint");
            }
        });

        register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                if (tardis.door().isOpen()) {
                    sendChat(player, class_2561.method_43470("Doors are already open"));
                    return failure(source);
                }

                sendChat(player, class_2561.method_43470("Opening TARDIS doors."));
                tardis.door().openDoors();
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("open", "open the door");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("open_door");
            }
        });

        register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                if (!tardis.door().isOpen()) {
                    sendChat(player, class_2561.method_43470("Doors are already closed"));
                    return failure(source);
                }

                sendChat(player, class_2561.method_43470("Closing TARDIS doors."));
                tardis.door().closeDoors();
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("close", "close the door");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("close_door");
            }
        });


        register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                TravelHandlerBase.State state = tardis.travel().getState();
                sendChat(player, class_2561.method_43470("TARDIS State: " + state.name()));

                if (state == TravelHandlerBase.State.FLIGHT) {
                    sendChat(player, class_2561.method_43470("Flight is " + tardis.travel().getDurationAsPercentage() + "% complete."));
                }

                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("progress", "flight status", "flight progress");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("progress");
            }
        });

        HandlesResponseRegistry.register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                sendChat(player, class_2561.method_43470("Toggled Shields."));
                tardis.shields().visuallyShielded().toggle();
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("toggle shields", "shields");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("toggle_shields");
            }
        });

        HandlesResponseRegistry.register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                if (tardis.isRefueling()) {
                    sendChat(player, class_2561.method_43470("Refueling is already enabled."));
                    return failure(source);
                }

                sendChat(player, class_2561.method_43470("Enabling Refueling."));
                tardis.travel().handbrake(true);
                tardis.setRefueling(true);
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("refuel");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("enable_refuel");
            }
        });

        HandlesResponseRegistry.register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                if (!tardis.isRefueling()) {
                    sendChat(player, class_2561.method_43470("Refueling is already disabled."));
                    return failure(source);
                }

                sendChat(player, class_2561.method_43470("Disabling Refueling."));
                tardis.travel().handbrake(false);
                tardis.setRefueling(false);
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("stop refuel");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("disable_refuel");
            }
        });

        HandlesResponseRegistry.register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                sendChat(player, class_2561.method_43470("Protocol 3 Toggled."));
                tardis.cloak().cloaked().toggle();
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("p3");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("toggle_cloak");
            }
        });

        HandlesResponseRegistry.register(new HandlesResponse() {
            @Override
            public boolean run(class_3222 player, HandlesSound source, ServerTardis tardis) {
                sendChat(player, class_2561.method_43470("Anti-Gravs Toggled."));
                tardis.travel().antigravs().toggle();
                return success(source);
            }

            @Override
            public List<String> getCommandWords() {
                return List.of("antigravs");
            }

            @Override
            public class_2960 id() {
                return AITMod.id("toggle_antigravs");
            }
        });
    }


    private static boolean onChatMessage(class_7471 signedMessage, class_3222 player, class_2556.class_7602 parameters) {
        class_1799 stack;
        String message = signedMessage.method_44862();

        boolean bl = message.toLowerCase().startsWith("handles");
        if (player.method_37908().method_8608()) return true;
        if (!bl) return true;

        String command = message.toLowerCase().replace(",", "")
                .replace("handles ", "");
        HandlesResponse response = get(command);

        for (int i = 0; i < player.method_31548().method_5439(); i++) {
            stack = player.method_31548().method_5438(i);

            if (stack.method_7909() instanceof HandlesItem item && item.isLinked(stack)) {
                Tardis tardis = item.getTardis(player.method_37908(), stack);

                if (tardis.butler().getHandles() == null) {
                    response.run(player, HandlesSound.of(player), tardis.asServer());
                    return false;
                }

                break;
            }
        }

        if (!(player.method_37908() instanceof TardisServerWorld tardisWorld))
            return true;

        Tardis tardis = tardisWorld.getTardis();

        if (tardis.butler().getHandles() == null)
            return true;

        if (response.requiresSudo() && tardis.stats().security().get()
                && !SecurityControl.hasMatchingKey(player, tardis))
            return true;

        response.run(player, HandlesSound.of(tardis.asServer()), tardis.asServer());
        return false;
    }
}
