/*
 * Copyright (C) 2025 AmbleLabs
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 * This code is MPL, due to it referencing this code: https://gitlab.com/cable-mc/cobblemon/-/blob/main/common/src/main/kotlin/com/cobblemon/mod/common/client/render/models/blockbench/bedrock/animation/BedrockAnimation.kt?ref_type=heads
 */


package dev.amble.lib.client.bedrock;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.annotations.SerializedName;

import dev.amble.lib.AmbleKit;
import dev.amble.lib.animation.AnimatedEntity;
import dev.amble.lib.animation.EffectProvider;
import dev.amble.lib.animation.client.AnimationMetadata;
import dev.amble.lib.animation.client.WorldPosition;
import lombok.AllArgsConstructor;
import lombok.RequiredArgsConstructor;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1109;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_243;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3414;
import net.minecraft.class_3545;
import net.minecraft.class_630;
import net.minecraft.class_7094;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static net.minecraft.class_3532.method_41303;


@AllArgsConstructor
@RequiredArgsConstructor
@Environment(EnvType.CLIENT)
public class BedrockAnimation {
	public static final Gson GSON = new GsonBuilder()
			.registerTypeAdapter(BedrockModel.LocatorBone.class, new BedrockModel.LocatorBone.Adapter())
			.registerTypeAdapter(BedrockAnimation.class, new BedrockAnimationAdapter())
			.create();

	public static boolean IS_RENDERING_PLAYER = false; // whether the fps camera is currently rendering the player
	public static boolean IS_RENDERING_HEAD = false; // whether the fps camera is currently rendering the player's head
	public static float HEAD_HIDE_DISTANCE = 0.5F; // If the camera is this close to the head it gets hidden
	public static Optional<Boolean> WAS_HUD_HIDDEN = Optional.empty(); // the state of the hud before starting an animation on the local player
	public static final Collection<String> IGNORED_BONES = Set.of("camera");
	public static final Collection<String> ROOT_BONES = Set.of("root", "player");


	public final boolean shouldLoop;
	public final double animationLength;
	public final Map<String, BoneTimeline> boneTimelines;
	public final boolean overrideBones;
	public final AnimationMetadata metadata;
	public final Map<Double, class_2960> sounds;
	public String name;

	@Nullable
	public static BedrockAnimation getFor(AnimatedEntity animated) {
		BedrockAnimationReference ref = animated.getCurrentAnimation();
		if (ref == null) return null;

		BedrockAnimation anim = ref.get().orElse(null);
		if (anim == null) return null;
		class_7094 state = animated.getAnimationState();
		if (state == null || anim.isFinished(state)) return null;

		return anim;
	}


	@Environment(EnvType.CLIENT)
	public void apply(class_630 root, double runningSeconds) {
		this.resetBones(root, this.overrideBones);

		this.boneTimelines.forEach((boneName, timeline) -> {
			try {
				if (IGNORED_BONES.contains(boneName.toLowerCase())) return;

				class_630 bone = root.method_32088().filter(part -> part.method_41919(boneName)).findFirst().map(part -> part.method_32086(boneName)).orElse(null);
				if (bone == null) {
					if (ROOT_BONES.contains(boneName.toLowerCase())) {
						bone = root;
					} else {
						throw new IllegalStateException("Bone " + boneName + " not found in model. If this is the root part, ensure it is named 'root'.");
					}
				}

				if (!timeline.position.isEmpty()) {
					class_243 position = timeline.position.resolve(runningSeconds);

					// traverse includes self
					bone.method_32088().forEach(child -> {
						child.field_3657 += (float) position.field_1352;
						child.field_3656 += (float) position.field_1351;
						child.field_3655 += (float) position.field_1350;
					});
				}

				if (!timeline.rotation.isEmpty()) {
					class_243 rotation = timeline.rotation.resolve(runningSeconds);

					bone.field_3654 += (float) Math.toRadians((float) rotation.field_1352);
					bone.field_3675 += (float) Math.toRadians((float) rotation.field_1351);
					bone.field_3674 += (float) Math.toRadians((float) rotation.field_1350);
				}

				if (!timeline.scale.isEmpty()) {
					class_243 scale = timeline.scale.resolve(runningSeconds);

					bone.method_32088().forEach(child -> {
						child.field_37938 = (float) scale.field_1352;
						child.field_37939 = (float) scale.field_1351;
						child.field_37940 = (float) scale.field_1350;
					});
				}
			} catch (Exception e) {
				///AmbleKit.LOGGER.error("Failed apply animation to {} in model. Skipping animation application for this bone.", boneName, e);
			}
		});

		boolean isComplete = !this.shouldLoop && runningSeconds >= this.animationLength;
		if (isComplete) {
			this.resetBones(root, true);
		}
	}

	public void applyEffects(@Nullable EffectProvider provider, double current, double previous, @Nullable class_630 root) {
		if (root != null) {
			// todo finish particles
			//WorldPosition.get(this, "right_arm", (float) current, provider, root).spawnParticle(ParticleTypes.FLAME, Vec3d.ZERO, 1);
			//WorldPosition.get(this, "left_arm", (float) current, provider, root).spawnParticle(ParticleTypes.FLAME, Vec3d.ZERO, 1);
			//WorldPosition.get(this, "head", (float) current, provider, root).spawnParticle(ParticleTypes.FLAME, Vec3d.ZERO, 1);
			//WorldPosition.get(this, "left_leg", (float) current, provider, root).spawnParticle(ParticleTypes.FLAME, Vec3d.ZERO, 1);
			//WorldPosition.get(this, "right_leg", (float) current, provider, root).spawnParticle(ParticleTypes.FLAME, Vec3d.ZERO, 1);
			//WorldPosition.get(this, "particle", (float) current, provider, root).spawnParticle(ParticleTypes.FLAME, Vec3d.ZERO, 1);
		}

		if (provider instanceof class_1297 entity) {
			if (!this.metadata.movement()) {
				entity.method_18799(class_243.field_1353);
				entity.field_6017 = 0;

				if (entity instanceof class_1309 living) {
					living.field_42108.method_48567(0F);
				}
			}
		}

		if (this.sounds == null || this.sounds.isEmpty()) return;

		for (Map.Entry<Double, class_2960> entry : this.sounds.entrySet()) {
			double time = entry.getKey();
			class_2960 soundId = entry.getValue();

			if (previous <= time && current >= time) {
				class_3414 event = class_3414.method_47908(soundId);

				if (provider != null) {
					if (!provider.isSilent()) {
						class_243 pos = provider.getEffectPosition(class_310.method_1551().method_1488());
						provider.getWorld().method_43128(class_310.method_1551().field_1724, pos.field_1352, pos.field_1351, pos.field_1350, event, provider.getSoundCategory(), 1F, 1F);
					}
				} else {
					class_310.method_1551().method_1483().method_4873(class_1109.method_4757(event, 1F, 1F));
				}
			}
		}
	}

	@Environment(EnvType.CLIENT)
	public void apply(class_630 root, class_7094 state, float progress, float speedMultiplier, @Nullable EffectProvider source) {
		double previous = getRunningSeconds(state);
		double seconds = getRunningSeconds(state, progress, speedMultiplier);
		state.method_41323(s -> {
			apply(root, seconds);
			applyEffects(source, seconds, previous, root);
		});
	}

	public void apply(class_630 root, int totalTicks, float rawDelta) {
		float ticks = (float) ((totalTicks / 20F) % (this.animationLength)) * 20;
		float delta = rawDelta / 10F;

		apply(root, (ticks / 20) + delta);
	}

	public double getRunningSeconds(class_7094 state, float progress, float speedMultiplier) {
		state.method_43686(progress, speedMultiplier);

		return getRunningSeconds(state);
	}

	public double getRunningSeconds(class_7094 state) {
		float f = (float)state.method_43687() / 1000.0F;
		double seconds = this.shouldLoop ? f % this.animationLength : f;

		return seconds;
	}

	public boolean isFinished(class_7094 state) {
		if (this.shouldLoop) return false;

		return getRunningSeconds(state) >= this.animationLength;
	}

	public void resetBones(class_630 root, boolean resetAll) {
		if (resetAll) {
			root.method_32088().forEach(class_630::method_41923);
			return;
		}

		this.boneTimelines.forEach((boneName, timeline) -> {
			try {
				if (IGNORED_BONES.contains(boneName.toLowerCase())) return;

				class_630 bone = root.method_32088().filter(part -> part.method_41919(boneName)).findFirst().map(part -> part.method_32086(boneName)).orElse(null);
				if (bone == null) {
					if (ROOT_BONES.contains(boneName.toLowerCase())) {
						bone = root;
					} else {
						throw new IllegalStateException("Bone " + boneName + " not found in model. If this is the root part, ensure it is named 'root'.");
					}
				}

				bone.method_32088().forEach(class_630::method_41923);
			} catch (Exception e) {
				//AmbleKit.LOGGER.error("Failed to reset animation on {} in model. Skipping animation reset for this bone.", boneName, e);
			}
		});
	}

	/**
	 * @return Pair<pitch, yaw> rotation for a given bone
	 */
	public class_3545<Float, Float> getRotations(String part, float progress) {
		if (!this.boneTimelines.containsKey(part)) return new class_3545<>(0F, 0F);

		class_243 rotation = this.boneTimelines.get(part).rotation().resolve(progress);

		return eulerToPitchYaw(rotation);
	}

	/**
	 * @return Pair<pitch, yaw> rotation for a given bone
	 */
	public static class_3545<Float, Float> eulerToPitchYaw(class_243 rotation) {
		double xRad = Math.toRadians(rotation.field_1352);
		double yRad = Math.toRadians(rotation.field_1351);
		double zRad = Math.toRadians(rotation.field_1350);

		class_243 vec = new class_243(0, 0, 1);

		// Rotate around X (pitch)
		vec = new class_243(
				vec.field_1352,
				vec.field_1351 * Math.cos(xRad) - vec.field_1350 * Math.sin(xRad),
				vec.field_1351 * Math.sin(xRad) + vec.field_1350 * Math.cos(xRad)
		);
		// Rotate around Y (yaw)
		vec = new class_243(
				vec.field_1352 * Math.cos(yRad) + vec.field_1350 * Math.sin(yRad),
				vec.field_1351,
				-vec.field_1352 * Math.sin(yRad) + vec.field_1350 * Math.cos(yRad)
		);
		// Rotate around Z (roll)
		vec = new class_243(
				vec.field_1352 * Math.cos(zRad) - vec.field_1351 * Math.sin(zRad),
				vec.field_1352 * Math.sin(zRad) + vec.field_1351 * Math.cos(zRad),
				vec.field_1350
		);

		float animYaw = (float) Math.toDegrees(Math.atan2(-vec.field_1352, vec.field_1350));
		float animPitch = (float) Math.toDegrees(Math.asin(-vec.field_1351 / vec.method_1033()));

		return new class_3545<>(animPitch, animYaw);
	}

	public static class Group {
		@SerializedName("format_version")
		public String version;
		public Map<String, BedrockAnimation> animations;
	}

	public record BoneTimeline(BoneValue position, BoneValue rotation, BoneValue scale) {
	}

	public static class SimpleBoneValue implements BoneValue {
		public final class_243 value;
		public final Transformation transformation;

		public SimpleBoneValue(class_243 value, Transformation transformation) {
			this.value = value.method_18805(1, (transformation == Transformation.POSITION) ? -1 : 1, 1);
			this.transformation = transformation;
		}

		@Override
		public class_243 resolve(double time) {
			return value;
		}

		@Override
		public boolean isEmpty() {
			return false;
		}
	}

	public static class KeyFrameBoneValue extends TreeMap<Double, KeyFrame> implements BoneValue {

		private KeyFrame getAtIndex(SortedMap<Double, KeyFrame> map, Integer index) {
			if (index == null) return null;
			if (index < 0 || index >= map.size()) return null;
			Double key = new ArrayList<>(map.keySet()).get(index);
			return map.get(key);
		}

		@Override
		public class_243 resolve(double time) {
			List<Double> keyList = new ArrayList<>(this.keySet());

			Integer afterIndex = null;
			for (int i = 0; i < keyList.size(); i++) {
				if (keyList.get(i) > time) {
					afterIndex = i;
					break;
				}
			}

			Integer beforeIndex;
			if (afterIndex == null) {
				beforeIndex = this.size() - 1;
			} else if (afterIndex == 0) {
				beforeIndex = null;
			} else {
				beforeIndex = afterIndex - 1;
			}

			KeyFrame after = getAtIndex(this, afterIndex);
			KeyFrame before = getAtIndex(this, beforeIndex);

			class_243 afterData = (after != null && after.getPre() != null) ? after.getPre().resolve(time) : class_243.field_1353;
			class_243 beforeData = (before != null && before.getPost() != null) ? before.getPost().resolve(time) : class_243.field_1353;

			if (before != null || after != null) {
				boolean smoothBefore = before != null && before.interpolationType == InterpolationType.SMOOTH;
				boolean smoothAfter = after != null && after.interpolationType == InterpolationType.SMOOTH;

				if (smoothBefore || smoothAfter) {
					if (before != null && after != null) {
						Integer beforePlusIndex = beforeIndex == 0 ? null : beforeIndex - 1;
						KeyFrame beforePlus = getAtIndex(this, beforePlusIndex);

						Integer afterPlusIndex = afterIndex == this.size() - 1 ? null : afterIndex + 1;
						KeyFrame afterPlus = getAtIndex(this, afterPlusIndex);

						class_243 beforePlusData = (beforePlus != null && beforePlus.getPost() != null) ? beforePlus.getPost().resolve(time) : beforeData;
						class_243 afterPlusData = (afterPlus != null && afterPlus.getPre() != null) ? afterPlus.getPre().resolve(time) : afterData;

						double t = (time - before.time) / (after.time - before.time);

						return new class_243(
								method_41303((float) t, (float) beforePlusData.field_1352, (float) beforeData.field_1352, (float) afterData.field_1352, (float) afterPlusData.field_1352),
								method_41303((float) t, (float) beforePlusData.field_1351, (float) beforeData.field_1351, (float) afterData.field_1351, (float) afterPlusData.field_1351),
								method_41303((float) t, (float) beforePlusData.field_1350, (float) beforeData.field_1350, (float) afterData.field_1350, (float) afterPlusData.field_1350)
						);
					} else if (before != null) {
						return beforeData;
					} else {
						return afterData;
					}
				} else {
					if (before != null && after != null) {
						double alpha = time;

						alpha = (alpha - before.time) / (after.time - before.time);

						return new class_243(
								beforeData.method_10216() + (afterData.method_10216() - beforeData.method_10216()) * alpha,
								beforeData.method_10214() + (afterData.method_10214() - beforeData.method_10214()) * alpha,
								beforeData.method_10215() + (afterData.method_10215() - beforeData.method_10215()) * alpha
						);
					} else if (before != null) {
						return beforeData;
					} else {
						return afterData;
					}
				}
			} else {
				return new class_243(0.0, 0.0, 0.0);
			}
		}
	}

	public static class EmptyBoneValue implements BoneValue {
		public static final EmptyBoneValue INSTANCE = new EmptyBoneValue();

		private EmptyBoneValue() {}

		@Override
		public class_243 resolve(double time) {
			return class_243.field_1353;
		}

		@Override
		public boolean isEmpty() {
			return true;
		}
	}


	public interface BoneValue {
		class_243 resolve(double time);
		boolean isEmpty();
	}

	public abstract static class KeyFrame {
		public final double time;
		public final Transformation transformation;
		public final InterpolationType interpolationType;

		public KeyFrame(double time, Transformation transformation, InterpolationType interpolationType) {
			this.time = time;
			this.transformation = transformation;
			this.interpolationType = interpolationType;
		}

		public abstract BoneValue getPre();
		public abstract BoneValue getPost();
	}

	public static class SimpleKeyFrame extends KeyFrame {
		public final BoneValue data;

		public SimpleKeyFrame(double time, Transformation transformation, InterpolationType interpolationType, BoneValue data) {
			super(time, transformation, interpolationType);
			this.data = data;
		}

		@Override
		public BoneValue getPre() {
			return data;
		}

		@Override
		public BoneValue getPost() {
			return data;
		}
	}

	public static class JumpKeyFrame extends KeyFrame {

		private final BoneValue pre;
		private final BoneValue post;

		public JumpKeyFrame(
				double time,
				Transformation transformation,
				InterpolationType interpolationType,
				BoneValue pre,
				BoneValue post
		) {
			super(time, transformation, interpolationType);
			this.pre = pre;
			this.post = post;
		}

		@Override
		public BoneValue getPre() {
			return pre;
		}

		@Override
		public BoneValue getPost() {
			return post;
		}
	}

	public enum InterpolationType {
		SMOOTH, LINEAR
	}

	public enum Transformation {
		POSITION, ROTATION, SCALE
	}
}
