/*
 * Copyright (C) 2025 AmbleLabs
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 * This code is MPL, due to it referencing this code: https://gitlab.com/cable-mc/cobblemon/-/blob/main/common/src/main/kotlin/com/cobblemon/mod/common/client/render/models/blockbench/bedrock/animation/BedrockAnimationRepository.kt
 */


package dev.amble.lib.client.bedrock;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import dev.amble.lib.AmbleKit;
import dev.amble.lib.register.Registry;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.resource.ResourceManagerHelper;
import net.fabricmc.fabric.api.resource.SimpleSynchronousResourceReloadListener;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import org.jetbrains.annotations.Nullable;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.Map;

@Environment(EnvType.CLIENT)
public class BedrockAnimationRegistry implements SimpleSynchronousResourceReloadListener, Registry {
	private static final BedrockAnimationRegistry INSTANCE = new BedrockAnimationRegistry();

	private final Map<String, BedrockAnimation.Group> groups = new HashMap<>();

	public BedrockAnimationRegistry() {
		ResourceManagerHelper.get(class_3264.field_14188).registerReloadListener(this);
	}

	public BedrockAnimation get(String fileName, String animationName) {
		BedrockAnimation.Group group = groups.get(fileName);
		if (group == null) {
			return null;
		}
		return group.animations.get(animationName);
	}

	public BedrockAnimation get(BedrockAnimationReference data) {
		return get(data.fileName(), data.animationName());
	}

	@Override
	public class_2960 getFabricId() {
		return AmbleKit.id("bedrock_animation");
	}

	@Override
	public void method_14491(class_3300 manager) {
		int animationCount = 0;
		groups.clear();

		for (class_2960 rawId : manager.method_14488("bedrock", filename -> filename.method_12832().endsWith(".animation.json")).keySet()) {
			try (InputStream stream = manager.method_14486(rawId).get().method_14482()) {
				JsonObject json = JsonParser.parseReader(new InputStreamReader(stream)).getAsJsonObject();

				@Nullable JsonObject metadata = null;
				class_2960 metadataId = class_2960.method_43902(rawId.method_12836(), rawId.method_12832().replaceFirst("\\.animation\\.json$", ".metadata.json"));
				if (manager.method_14486(metadataId).isPresent()) {
					try (InputStream metaStream = manager.method_14486(metadataId).get().method_14482()) {
						metadata = JsonParser.parseReader(new InputStreamReader(metaStream)).getAsJsonObject();
					} catch (Exception e) {
						AmbleKit.LOGGER.error("Error occurred while loading metadata for bedrock model {}", rawId.toString(), e);
					}
				}

				if (metadata != null) {
					// each key will be anim id
					for (String key : metadata.keySet()) {
						if (json.has("animations") && json.getAsJsonObject("animations").has(key)) {
							json.getAsJsonObject("animations").getAsJsonObject(key).add("metadata", metadata.getAsJsonObject(key));
						}
					}
				}

				BedrockAnimation.Group group = BedrockAnimation.GSON.fromJson(json, BedrockAnimation.Group.class);

				group.animations.forEach((name, animation) -> animation.name = name);

				String groupName = rawId.method_12832().substring(rawId.method_12832().lastIndexOf("/") + 1).replace(".animation.json", "");
				groups.put(groupName, group);
				animationCount += group.animations.size();
			} catch (Exception e) {
				AmbleKit.LOGGER.error("Error occurred while loading resource json {}", rawId.toString(), e);
			}
		}

		AmbleKit.LOGGER.info("Loaded {} animations from {} groups", animationCount, groups.size());
	}

	public static BedrockAnimationRegistry getInstance() {
		return INSTANCE;
	}
}
