package dev.amble.lib.datagen.sound;

import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import net.fabricmc.fabric.api.datagen.v1.FabricDataOutput;
import net.minecraft.class_2405;
import net.minecraft.class_3414;
import net.minecraft.class_7403;
import net.minecraft.class_7784;
import net.minecraft.class_7923;
import org.jetbrains.annotations.Nullable;
import dev.amble.lib.AmbleKit;
import dev.amble.lib.util.StringCursor;

/**
 * Datagen Provider for sounds, this class is used to generate the sounds.json file for the mod
 */
public class AmbleSoundProvider implements class_2405 {

    protected final FabricDataOutput dataOutput;
    private final Map<String, Set<SoundEventWrapper>> sounds = new HashMap<>();
    private final boolean extractVariants;

    public AmbleSoundProvider(FabricDataOutput dataOutput) {
        this(dataOutput, true);
    }

    public AmbleSoundProvider(FabricDataOutput dataOutput, boolean extractVariants) {
        this.dataOutput = dataOutput;
        this.extractVariants = extractVariants;
    }

    private boolean canAdd(String name, boolean check) {
        // you can't have duplicates in registries, nor bad ids.
        return check && !checkName(name);
    }

    public void addSound(String name, class_3414 event) {
        this.addSound(name, true, event);
    }

    public void addSound(String name, boolean check, class_3414 event) {
        if (canAdd(name, check))
            return;

        sounds.computeIfAbsent(name, s -> new HashSet<>()).add(new SoundEventWrapper(event));
    }

    public void addSound(String name, class_3414... events) {
        this.addSound(name, true, events);
    }

    public void addSound(String name, boolean check, class_3414... events) {
        if (canAdd(name, check))
            return;

        Set<SoundEventWrapper> set = sounds.computeIfAbsent(name, s -> new HashSet<>());

        for (class_3414 event : events) {
            set.add(new SoundEventWrapper(event));
        }
    }

    @Override
    public CompletableFuture<?> method_10319(class_7403 writer) {
        getSoundsFromMod(dataOutput.getModId()).forEach(sound -> {
            String path = sound.method_14833().method_12832();

            addSound(path, false, sound);

            if (extractVariants) {
                String newPath = extractPath(path);

                // event is not a variant!
                if (newPath == null)
                    return;

                addSound(newPath, false, sound);
            }
        });

        JsonObject soundsJson = new JsonObject();
        sounds.forEach((soundName, soundEvents) ->
                soundsJson.add(soundName, serializeSounds(soundEvents)));

        return class_2405.method_10320(writer, soundsJson, getOutputPath());
    }

    public Path getOutputPath() {
        return dataOutput.method_45972(class_7784.class_7490.field_39368).resolve(dataOutput.getModId())
                .resolve("sounds.json");
    }

    @Override
    public String method_10321() {
        return "Sound Definitions";
    }

    private static JsonObject serializeSounds(Iterable<SoundEventWrapper> wrappers) {
        JsonObject obj = new JsonObject();
        JsonArray sounds = new JsonArray();

        //tardis/moody/moody,tardis/moody/moody1,tardis/moody/moody2 =>
        //tardis/moody/moody:[tardis/moody/moody, tardis/moody/moody1, tardis/moody/moody2]
        for (SoundEventWrapper wrapper : wrappers) {
            sounds.add(wrapper.event.method_14833().toString());
        }

        obj.add("sounds", sounds);
        return obj;
    }

    @Nullable private static String extractPath(String path) {
        StringCursor cursor = new StringCursor(path, path.length() - 1, -1);

        while (true) {
            if (!Character.isDigit(cursor.peek()))
                return cursor.substring();

            cursor.next();
        }
    }

    private static boolean checkName(String name) {
        if (name.contains(" ")) {
            AmbleKit.LOGGER.error("Sound event name cannot contain spaces: {}", name);
            return false;
        }

        for (Character character : name.toCharArray()) {
            if (Character.isUpperCase(character)) {
                AmbleKit.LOGGER.error("Sound event name cannot contain capital letters: {}", name);
                return false;
            }
        }

        return true;
    }

    public static Stream<class_3414> getSoundsFromMod(String namespace) {
        return class_7923.field_41172.method_10220().filter(sound -> sound.method_14833().method_12836().equals(namespace));
    }

    static class SoundEventWrapper {

        private final class_3414 event;

        public SoundEventWrapper(class_3414 event) {
            this.event = event;
        }

        @Override
        public int hashCode() {
            return event.method_14833().hashCode();
        }
    }
}
