package dev.amble.lib.skin;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import dev.amble.lib.AmbleKit;
import dev.amble.lib.util.ServerLifecycleHooks;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayConnectionEvents;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayNetworking;
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerLifecycleEvents;
import net.fabricmc.fabric.api.networking.v1.PacketByteBufs;
import net.fabricmc.fabric.api.networking.v1.ServerPlayConnectionEvents;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.class_5218;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.Nullable;

import java.io.Reader;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

public class SkinTracker extends HashMap<UUID, SkinData> {
	public static final class_2960 SYNC_KEY = AmbleKit.id("skin_sync");

	private static SkinTracker INSTANCE;

	public static SkinTracker getInstance() {
		if (INSTANCE == null) {
			INSTANCE = new SkinTracker();
		}
		return INSTANCE;
	}

	public static void init() {
		INSTANCE = new SkinTracker();

		ServerPlayConnectionEvents.JOIN.register((handler, sender, server) -> {
			getInstance().sync(handler.method_32311());
		});

		ServerLifecycleEvents.SERVER_STOPPING.register(server -> {
			getInstance().write(server);
		});

		ServerLifecycleEvents.SERVER_STARTED.register(SkinTracker::read);

		if (FabricLoader.getInstance().getEnvironmentType() == EnvType.CLIENT) {
			initClient();
		}
	}

	@Environment(EnvType.CLIENT)
	private static void initClient() {
		ClientPlayConnectionEvents.DISCONNECT.register((handler, client) -> {
			// getInstance().clear();
		});

		ClientPlayNetworking.registerGlobalReceiver(SYNC_KEY, ((client, handler, buf, responseSender) -> {
			getInstance().receive(buf);
		}));
	}

	@Nullable
	public SkinData putSynced(UUID id, SkinData data) {
		SkinData previous = this.put(id, data);
		sync(toBuf(id, data));
		return previous;
	}

	@Nullable
	public SkinData removeSynced(UUID id) {
		SkinData previous = this.remove(id);
		sync(toBuf(id, SkinData.clear()));
		return previous;
	}

	public Optional<SkinData> getOptional(UUID id) {
		return Optional.ofNullable(this.get(id));
	}

	private class_2540 toBuf(UUID id, SkinData data) {
		class_2540 buf = PacketByteBufs.create();

		buf.writeInt(1);

		buf.method_10797(id);
		data.writeBuf(buf);

		return buf;
	}

	private class_2540 toBuf() {
		return toBuf(this);
	}

	private class_2540 toBuf(Map<UUID, SkinData> map) {
		class_2540 buf = PacketByteBufs.create();

		buf.writeInt(map.size());

		for (Map.Entry<UUID, SkinData> entry : map.entrySet()) {
			buf.method_10797(entry.getKey());

			entry.getValue().writeBuf(buf);
		}

		return buf;
	}

	private void sync(class_2540 buf) {
		ServerLifecycleHooks.get().method_3760().method_14571().forEach((p) -> this.sync(buf, p));
	}

	private void sync(class_2540 buf, class_3222 player) {
		ServerPlayNetworking.send(player, SYNC_KEY, buf);
	}

	private void receive(class_2540 buf) {
		int count = buf.readInt();
		for (int i = 0; i < count; i++) {
			UUID id = buf.method_10790();
			SkinData val = SkinData.readBuf(buf);
			if (val == null) continue;
			this.put(id, val);
		}
	}

	public void sync() {
		sync(toBuf());
	}

	public void sync(class_3222 target) {
		sync(toBuf(), target);
	}

	private static Path getSavePath(MinecraftServer server) {
		return server.method_27050(class_5218.field_24188).resolve("amblekit").resolve("skins.json");
	}

	private void write(MinecraftServer server) {
		try {
			Path savePath = getSavePath(server);
			if (!Files.exists(savePath)) {
				Files.createDirectories(savePath.getParent());
			}

			Files.writeString(savePath, AmbleKit.GSON.toJson(this, SkinTracker.class));
		} catch (Exception e) {
			AmbleKit.LOGGER.error("Failed to write skins.json", e);
		}
	}

	private static void read(MinecraftServer server) {
		if (!(Files.exists(getSavePath(server)))) return;

		try {
			String raw = Files.readString(getSavePath(server));
			JsonObject object = JsonParser.parseString(raw).getAsJsonObject();
			INSTANCE = AmbleKit.GSON.fromJson(object, SkinTracker.class);
			INSTANCE.sync();
		} catch (Exception e) {
			AmbleKit.LOGGER.error("Failed to read skins.json", e);
		}
	}
}
